//******************************************************************************
//   MSP-FET430P140 Demo - SMBus Master reads from SMBus MSP430 Slave
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. The master 
//   reads from the slave. This is the master code. The slave code is called
//   fet140_SMB_slav_slvrst.c. The data from the slave transmitter begins
//   at 1 and is incremented with each transfer. The RXRDYIFG interrupt is used to 
//   know when new data has been received.
//   ACLK = n/a, MCLK = SMCLK = I2CIN = DCO ~ 800kHz
//   //* MSP430F169 Device Required *//
//   
//                                 /|\  /|\
//                  MSP430F169     10k  10k     MSP430F169
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1             |
//             |                 |  |      |             P1.0|-->LED
//             |                 |  |      |                 |
//    P4.0(TB0)|<----------- P3.3|<-+----->|P3.3             |
//             |                 |         |                 |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  Feb 2005
//  Built with IAR Embedded Workbench Version: 3.21A
//******************************************************************************

#include  <msp430x16x.h>

typedef enum {
             SMBS_MISC               = (unsigned char) 0x48,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x0C,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);  

char RXData = 0;
char TestData = 0;

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P3SEL |= 0x0A;                            // Select I2C pins
  P1DIR |= 0xFF;                            // P1.0 output
                                            // unused ports to output
  P2DIR =  0xFF;
  P3DIR =  0xFF;
  P4DIR =  0xFF;
  P5DIR =  0xFF;
  P6DIR =  0xFF;
  P1OUT =  0;
  SMBS_Init (SMBS_MISC,0x48);               // Slave address configured as 48h
  while (1)
  {
    U0CTL |= MST;                           // Master mode
    I2CTCTL |= I2CSTT + I2CSTP;             // Initiate transfer       
    _BIS_SR(CPUOFF+GIE);                    // Enter LPM0, Enable interrupts
  } 

}

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  U0CTL |= I2C + SYNC;                      // Recommended init procedure
  U0CTL &= ~I2CEN;                          // Recommended init procedure
  I2CTCTL |= I2CSSEL1;                      // SMCLK
  I2CNDAT = 0x01;                           // Read one byte
  U0CTL |= I2CEN;                           // Enable I2C
  if (Add_Param == SMBS_MISC)
  {
      I2CSA = Slave_Add;                    // Slave Address is 048h
  } 
  else
      I2CSA = (unsigned char) Add_Param ;
  
  I2CIE = RXRDYIE;                          // Enable RXRDYIFG interrupt
  U0CTL |= I2CEN;                           // Enable I2C
}

// Common ISR for I2C Module
#pragma vector=USART0TX_VECTOR
__interrupt void I2C_ISR(void)
{
 
 switch(I2CIV)
 {
   case  0: break;                          // No interrupt
   case  2: break;                          // Arbitration lost
   case  4: break;                          // No Acknowledge
   case  6: break;                          // Own Address
   case  8: break;                          // Register Access Ready
   case 10:                                 // Receive Ready
     RXData = I2CDRB;                       // RX data
     _BIC_SR_IRQ(CPUOFF);                   // Clear LPM0
     break; 
   case 12: break;                          // Transmit Ready
   case 14: break;                          // General Call
   case 16: break;                          // Start Condition
 }
} 
